----------------------------------------------------------------------------
-- @Author: JTSage / FSG Modding -------------------------------------------
----------------------------------------------------------------------------

MoreDustExtension = {}

local MoreDustExtension_mt = Class(MoreDustExtension)

MoreDustExtension.TURN_OFF_EXHAUST = true
MoreDustExtension.MAX_USER_SCALE   = 3
MoreDustExtension.VEHICLE_SCALE    = {
	BALER      = 1.25,
	CLEANING   = 1.75,
	COMBINE    = 1.25,
	CULTIVATOR = 2.5,
	CUTTER     = 1.15,
	MOWER      = 1.1,
	MULCHER    = 1.75,
	PLOW       = 1.25,
	ROLLER     = 1.75,
	SOWING     = 2,
	WEEDER     = 1.5,
	WHEELS     = 1.1,
	WINDROW    = 1.3
}

MoreDustExtension.VEHICLE_L10N = {
	BALER      = {"category_balers", 1.25},
	CLEANING   = {"menu_MoreDustExtension_potatobeet", 1.75},
	COMBINE    = {"category_harvesters", 1.25},
	CULTIVATOR = {"category_cultivators", 2.5},
	CUTTER     = {"category_cutters", 1.15},
	MOWER      = {"category_mowers", 1.1},
	MULCHER    = {"category_mulchers", 1.75},
	PLOW       = {"category_plows", 1.25},
	ROLLER     = {"category_rollers", 1.75},
	SOWING     = {"category_sowingMachines", 2},
	WEEDER     = {"category_weeders", 1.5},
	WHEELS     = {"configuration_wheelSetup", 1.1},
	WINDROW    = {"category_windrowers", 1.3},
}

MoreDustExtension.VEHICLE_ORDER = {
	"BALER", "CLEANING", "COMBINE", "CULTIVATOR", "CUTTER", "MOWER", "MULCHER",
	"PLOW", "ROLLER", "SOWING", "WEEDER", "WHEELS", "WINDROW"
}

MoreDustExtension.MENU_FRACTIONAL = {
	"0.1", "0.2", "0.3", "0.4", "0.5", "0.6", "0.7", "0.8", "0.9", "1.0",
	"1.1", "1.2", "1.3", "1.4", "1.5", "1.6", "1.7", "1.8", "1.9", "2.0",
	"2.1", "2.2", "2.3", "2.4", "2.5", "2.6", "2.7", "2.8", "2.9", "3.0"
}

MoreDustExtension.MENU_PERCENT = {
	"5 %", "10 %", "15 %", "20 %", "25 %", "30 %", "35 %", "40 %", "45 %", "50 %",
	"55 %", "60 %", "65 %", "70 %", "75 %", "80 %", "85 %", "90 %", "95 %", "100 %",
	"105 %", "110 %", "115 %", "120 %", "125 %", "130 %", "135 %", "140 %", "145 %", "150 %",
	"155 %", "160 %", "165 %", "170 %", "175 %", "180 %", "185 %", "190 %", "195 %", "200 %",
	"205 %", "210 %", "215 %", "220 %", "225 %", "230 %", "235 %", "240 %", "245 %", "250 %",
	"255 %", "260 %", "265 %", "270 %", "275 %", "280 %", "285 %", "290 %", "295 %", "300 %",
}



function MoreDustExtension:new(mission, i18n, modDirectory, modName)
	local self = setmetatable({}, MoreDustExtension_mt)

	self.myName            = "MoreDustExtension"
	self.isServer          = mission:getIsServer()
	self.isClient          = mission:getIsClient()
	self.mission           = mission
	self.i18n              = i18n
	self.modDirectory      = modDirectory
	self.modName           = modName
	self.userScale         = 1

	return self
end


function MoreDustExtension.getScaled(original, vehicleTypeScale, min, wheelSystem)
	local currentUserScale = getfenv(0)["g_moreDustExtension"]["userScale"]

	if wheelSystem and currentUserScale > 1 then
		if currentUserScale > 2 then
			currentUserScale = 1.5 + ( (currentUserScale - 2) * 0.25)
		else
			currentUserScale = 1 + ( (currentUserScale - 1) * 0.5)
		end
	end

	return MoreDustExtension.timeScale(
		g_currentMission.environment.weather:getTimeSinceLastRain(),
		original,
		original * vehicleTypeScale,
		currentUserScale,
		min
	)
end


function MoreDustExtension.timeScale(time, standard, max, user, min)
	local TIME_MIN   = 10
	local TIME_STD   = 30
	local TIME_MAX   = 120
	local MIN_RETURN = min or 1

	local cleanReturn = standard

	if time < TIME_MIN then
		cleanReturn = standard * 0.01
	elseif time < TIME_STD then
		cleanReturn = ((time / TIME_STD) * standard) * user
	elseif time < TIME_MAX then
		cleanReturn = math.max(standard, ((time / TIME_MAX) * max)) * user
	else
		cleanReturn = max * user
	end

	if cleanReturn < MIN_RETURN then
		cleanReturn = MIN_RETURN
	end

	return cleanReturn
end


function MoreDustExtension:onUpdateTickWheels(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	local spec = self.spec_wheels

	if self.isClient and self:getLastSpeed() > 1 then
		for _, wheel in pairs(spec.wheels) do
			if wheel.driveGroundParticleSystems ~= nil and wheel.driveGroundParticleSystems.wheel_dust then
				for _, partSys in ipairs(wheel.driveGroundParticleSystems.wheel_dust) do
					if partSys.isEmitting then
						MoreDustExtension:lifespanByName( partSys, "WHEELS", true)
					end
				end
			end
		end
	end
end


function MoreDustExtension:onUpdateTickWP(superFunc, dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	-- workParticles version (roller, plow/cultivator, sowing machine/planter, weeder)
	local spec          = self.spec_workParticles
	local isOnField     = self:getIsOnField()
	local lastSpeed     = self:getLastSpeed(true)
	local enabled       = self:getDoGroundManipulation() and isOnField
	local thisVehType

	if self.spec_plow ~= nil then 
		thisVehType = "PLOW"
	elseif self.spec_cultivator ~= nil then
		thisVehType = "CULTIVATOR"
	elseif self.spec_roller ~= nil then
		thisVehType = "ROLLER"
	elseif self.spec_sowingMachine ~= nil then
		thisVehType = "SOWING"
	elseif self.spec_weeder ~= nil then
		thisVehType = "WEEDER"
	end

	if thisVehType ~= nil then
		for _, ps in pairs(spec.particles) do
			for _, mapping in ipairs(ps.mappings) do
				local nodeEnabled = enabled

				nodeEnabled = nodeEnabled and mapping.groundRefNode.isActive and mapping.speedThreshold < lastSpeed

				if mapping.movingDirection ~= nil then
					nodeEnabled = nodeEnabled and mapping.movingDirection == self.movingDirection
				end

				if (nodeEnabled) then
					MoreDustExtension:lifespanByName(
						mapping.particleSystem,
						thisVehType
					)
					MoreDustExtension:cutEmitCount(mapping.particleSystem)
				end
			end
		end
	end

	superFunc(self, dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
end


function MoreDustExtension:onUpdateTickCutter(superFunc, dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	-- Cutter heads
	local spec = self.spec_cutter

	if self.isClient and spec.isWorking then
		MoreDustExtension:safeUpdateChildSystems(spec.cutterEffects, "CUTTER", false, false)
	end

	superFunc(self, dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
end


function MoreDustExtension:onUpdateTickCombi(superFunc, dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	-- Combines
	local spec = self.spec_combine

	if self.isClient then
		-- So far as I can tell, potato harvesters only.
		for _, fillEffect in ipairs(spec.fillEffects) do
			if fillEffect ~= nil and type(fillEffect) == "table" then
				if fillEffect.particleType == "CLEANING_DUST" then
					MoreDustExtension:lifespanByName(
						fillEffect.particleSystem,
						"CLEANING"
					)
					MoreDustExtension:cutEmitCount(fillEffect.particleSystem)
				end
			end
		end

		if spec.chopperPSenabled then
			-- Chopper is active, do something with it
			MoreDustExtension:safeUpdateChildSystems(spec.chopperEffects, "COMBINE", false, false)
		end

		if spec.strawPSenabled then
			-- straw is active, do something with it
			MoreDustExtension:safeUpdateChildSystems(spec.strawEffects, "COMBINE", false, false)
		end
	end

	superFunc(self, dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
end


function MoreDustExtension:onUpdateTickBaler(superFunc, dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	-- Balers
	local spec = self.spec_baler

	if self.isClient and self:getIsTurnedOn() and spec.lastAreaBiggerZeroTime > 0 then
		MoreDustExtension:safeUpdateChildSystems(spec.fillEffects, "BALER", true, true)
	end

	superFunc(self, dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
end


function MoreDustExtension:onUpdateTickMulch(superFunc, dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	-- Mulchers
	local spec = self.spec_mulcher

	if self.isClient and spec.isWorking then
		MoreDustExtension:safeUpdateParentSystem(spec.effects, "MULCHER", true, true)
	end

	superFunc(self, dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
end


function MoreDustExtension:onUpdateTickWind(superFunc, dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	-- Windrowers
	local spec = self.spec_windrower

	if self.isClient and spec.isWorking then
		MoreDustExtension:safeUpdateParentSystem(spec.effects, "WINDROW", false, false)
	end

	superFunc(self, dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
end

function MoreDustExtension:doMower(superFunc, dt)
	-- Mowers
	local spec = self.spec_mower

	if self.isClient and spec.isWorking then
		MoreDustExtension:safeUpdateParentSystem(spec.dropEffects, "MOWER", true, true)
	end

	superFunc(self, dt)
end

function MoreDustExtension:onUpdateTickMotorized(superFunc, dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	local spec = self.spec_motorized

	if self.isClient and MoreDustExtension.TURN_OFF_EXHAUST and not g_modIsLoaded["FS22_exhaustExtension"] then
		if self:getIsMotorStarted() then
			if spec.exhaustParticleSystems ~= nil then
				for _, ps in pairs(spec.exhaustParticleSystems) do
					ParticleUtil.setEmitCountScale(spec.exhaustParticleSystems, 0.1)
					ParticleUtil.setParticleLifespan(ps, 1)
				end
			end
		end
	end
end

function MoreDustExtension:safeUpdateParentSystem(parentEffectTable, vehType, alterScale, alterTrans)
	-- Update with lots of nil checks a parent Effect Table (recursive)
	if parentEffectTable ~= nil and type(parentEffectTable) == "table" then
		for _, effectTable in ipairs(parentEffectTable) do
			if effectTable.isActive then
				MoreDustExtension:safeUpdateChildSystems(effectTable.effects, vehType, alterScale, alterTrans)
			end
		end
	end
end

function MoreDustExtension:safeUpdateChildSystems(childEffects, vehType, alterScale, alterTrans)
	-- Update with lots of nil checks an effect table
	if childEffects ~= nil and type(childEffects) == "table" then
		for _, effectSys in ipairs(childEffects) do
			if effectSys.materialType == "smokeParticle" then
				MoreDustExtension:lifespanByName(
					effectSys.particleSystem,
					vehType
				)
				MoreDustExtension:cutEmitCount(effectSys.particleSystem)

				MoreDustExtension:safeAlterShape(effectSys, alterScale, alterTrans)
			end
		end
	end
end


function MoreDustExtension:safeAlterShape(effectSys, alterScale, alterTrans)
	-- double the length scale, move the translation back a bit
	--   (on demand, dependant on implement class)
	if alterScale and effectSys.emitterShape ~= nil and g_updateLoopIndex % 10 == 0 then
		local sx, sy, sz = getScale(effectSys.emitterShape)
		if effectSys.MDE_orig_scale_z == nil then
			effectSys.MDE_orig_scale_z = sz
		end
		if sz ~= nil and sz == effectSys.MDE_orig_scale_z then
			setScale(effectSys.emitterShape, sx, sy, sz * 2)
		end
	end

	if alterTrans and effectSys.emitterShape ~= nil and g_updateLoopIndex % 10 == 0 then
		local x, y, z = getTranslation(effectSys.emitterShape)
		if effectSys.MDE_orig_trans_z == nil then
			effectSys.MDE_orig_trans_z = z
		end
		if z ~= nil and z == effectSys.MDE_orig_trans_z then
			setTranslation(effectSys.emitterShape, x, y, (z-0.5))
		end
	end
end

function MoreDustExtension:cutEmitCount(partSys)
	-- Cut emitCountScale down when user scale > 1
	if  partSys == nil or partSys.emitCountScale == nil then
		return
	end

	if partSys.originalEmitScale == nil then
		partSys.originalEmitScale = partSys.emitCountScale
		partSys.currentEmitScale  = partSys.emitCountScale
	end

	if getfenv(0)["g_moreDustExtension"]["userScale"] < 1.1 then
		if partSys.currentEmitScale ~= partSys.originalEmitScale then
			partSys.currentEmitScale = partSys.originalEmitScale
			ParticleUtil.setEmitCountScale(partSys, partSys.originalEmitScale)
		end
	else
		local emitScaleFactor = 1 - (((getfenv(0)["g_moreDustExtension"]["userScale"] - 1) / 10) * 0.8)
		local newEmitScale    = emitScaleFactor * partSys.originalEmitScale

		if partSys.currentEmitScale ~= newEmitScale then
			partSys.currentEmitScale = newEmitScale
			ParticleUtil.setEmitCountScale(partSys, newEmitScale)
		end
	end
end


function MoreDustExtension:lifespanByName(partSys, name, wheelSystem)
	-- Alter dust particle lifespan with named scale (and user scale)
	if partSys == nil or partSys.originalLifespan == nil then
		return
	end

	if partSys.originalSetLifespan == nil then
		partSys.originalSetLifespan = getParticleSystemLifespan(partSys.geometry)
		partSys.currentSetLifespan  = getParticleSystemLifespan(partSys.geometry)
	end

	local newLifespan = math.ceil(MoreDustExtension.getScaled(
		partSys.originalSetLifespan,
		MoreDustExtension.VEHICLE_SCALE[name],
		100,
		wheelSystem
	))

	-- print("orig: " .. partSys.originalSetLifespan .. "  new: " .. newLifespan)

	if partSys.currentSetLifespan ~= newLifespan then
		partSys.currentSetLifespan = newLifespan
		ParticleUtil.setParticleLifespan(partSys, newLifespan)
	end
end


function MoreDustExtension:registerActionEvents()
	local _, actionEventId = g_inputBinding:registerActionEvent(
		"MoreDustExtension_user_scale", self,
		MoreDustExtension.actionUserScale,
		false, true, false, true, nil
	)
	MoreDustExtension.actionEventId = actionEventId

	MoreDustExtension:updateHelp()
	g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_HIGH)

	local _, actionEventIdEE = g_inputBinding:registerActionEvent(
		"MoreDustExtension_toggle_exhaust", self,
		MoreDustExtension.actionToggleExhaust,
		false, true, false, true, nil
	)
	g_inputBinding:setActionEventTextVisibility(actionEventIdEE, false)
end

function MoreDustExtension:updateHelp()
	local thisModEnviroment = getfenv(0)["g_moreDustExtension"]

	g_inputBinding:setActionEventText(
		MoreDustExtension.actionEventId,
		("%s (x%.1f)"):format(
			g_i18n:getText("input_MoreDustExtension_user_scale"),
			thisModEnviroment.userScale
		)
	)
	g_inputBinding:setActionEventTextVisibility(MoreDustExtension.actionEventId, true)
end


function MoreDustExtension:actionToggleExhaust()
	local thisModEnviroment = getfenv(0)["g_moreDustExtension"]

	thisModEnviroment.TURN_OFF_EXHAUST = not thisModEnviroment.TURN_OFF_EXHAUST

	local currentSettingString = g_i18n:getText("input_MoreDustExtension_toggle_exhaust") .. " - "

	if thisModEnviroment.TURN_OFF_EXHAUST then
		currentSettingString = currentSettingString .. g_i18n:getText("ui_off")
	else
		currentSettingString = currentSettingString .. g_i18n:getText("ui_on")
	end

	g_currentMission.hud:addSideNotification(
		FSBaseMission.INGAME_NOTIFICATION_OK,
		currentSettingString,
		2000
	)

	thisModEnviroment:saveSettings()
end


function MoreDustExtension:actionUserScale(_, inputValue)
	local thisModEnviroment = getfenv(0)["g_moreDustExtension"]

	thisModEnviroment.userScale = MathUtil.clamp(
		thisModEnviroment.userScale + ( 0.1 * inputValue ),
		0.1,
		thisModEnviroment.MAX_USER_SCALE
	)

	thisModEnviroment:updateHelp()
	thisModEnviroment:saveSettings()
end


function MoreDustExtension:saveSettings()
	local savegameFolderPath = ('%smodSettings/FS22_DustyLandsExtension/savegame%d'):format(getUserProfileAppPath(), g_currentMission.missionInfo.savegameIndex)
	local savegameFile       = savegameFolderPath .. "/moreDustExtension.xml"
	local thisModEnviroment  = getfenv(0)["g_moreDustExtension"]

	if ( not fileExists(savegameFile) ) then
		createFolder(('%smodSettings/FS22_DustyLandsExtension'):format(getUserProfileAppPath()))
		createFolder(savegameFolderPath)
	end

	local key = "moreDustExtension"
	local xmlFile = createXMLFile(key, savegameFile, key)

	setXMLFloat(xmlFile, key .. ".userScale#value", thisModEnviroment["userScale"])
	setXMLBool(xmlFile, key .. ".turnOffExhaust#value", thisModEnviroment.TURN_OFF_EXHAUST)

	for name,value in pairs(thisModEnviroment.VEHICLE_SCALE) do
		setXMLFloat(xmlFile, key .. "." .. name .. "#scale", value)
	end

	saveXMLFile(xmlFile)
	print("~~DustyLandsExtension :: saved config file")
end


function MoreDustExtension:loadSettings()
	local savegameFolderPath = ('%smodSettings/FS22_DustyLandsExtension/savegame%d'):format(getUserProfileAppPath(), g_currentMission.missionInfo.savegameIndex)
	local key                = "moreDustExtension"
	local thisModEnviroment  = getfenv(0)["g_moreDustExtension"]

	if fileExists(savegameFolderPath .. "/moreDustExtension.xml") then
		print("~~DustyLandsExtension :: loading config file")
		local xmlFile                      = loadXMLFile(key, savegameFolderPath .. "/moreDustExtension.xml")
		local savedUserScale               = Utils.getNoNil(getXMLFloat(xmlFile, key .. ".userScale#value"), 1)
		thisModEnviroment.TURN_OFF_EXHAUST = Utils.getNoNil(getXMLBool(xmlFile, key .. ".turnOffExhaust#value"), true)
		thisModEnviroment.userScale        = MathUtil.clamp(
			(math.floor(savedUserScale * 10)) / 10,
			0.1,
			thisModEnviroment.MAX_USER_SCALE
		)

		for name,value in pairs(thisModEnviroment.VEHICLE_SCALE) do
			thisModEnviroment.VEHICLE_SCALE[name] = Utils.getNoNil(
				getXMLFloat(xmlFile, key ..  "." .. name .. "#scale"),
				value
			)
		end

		delete(xmlFile)
	end
end


function MoreDustExtension.initGui(self)
	if not g_moreDustExtension.createdGUI then -- Skip if we've already done this once
		g_moreDustExtension.createdGUI = true

		self.menuOption_UserScale = self.checkInvertYLook:clone()
		self.menuOption_UserScale.target = g_moreDustExtension
		self.menuOption_UserScale.id = "moreDustExtension_UserScale"
		self.menuOption_UserScale:setCallback("onClickCallback", "onMenuOptionChanged_ScaleChangeUser")
		self.menuOption_UserScale:setDisabled(false)

		local settingTitle = self.menuOption_UserScale.elements[4]
		local toolTip      = self.menuOption_UserScale.elements[6]

		self.menuOption_UserScale:setTexts(MoreDustExtension.MENU_FRACTIONAL)

		settingTitle:setText(g_i18n:getText("input_MoreDustExtension_user_scale"))
		toolTip:setText(g_i18n:getText("menu_MoreDustExtension_tooltip_user_scale"))


		self.menuOption_ToggleExhaust = self.checkInvertYLook:clone()
		self.menuOption_ToggleExhaust.target = g_moreDustExtension
		self.menuOption_ToggleExhaust.id = "moreDustExtension_UserScale"
		self.menuOption_ToggleExhaust:setCallback("onClickCallback", "onMenuOptionChanged_toggleExhaust")
		self.menuOption_ToggleExhaust:setDisabled(false)

		local settingTitle = self.menuOption_ToggleExhaust.elements[4]
		local toolTip      = self.menuOption_ToggleExhaust.elements[6]

		self.menuOption_ToggleExhaust:setTexts({g_i18n:getText("ui_off"), g_i18n:getText("ui_on")})

		settingTitle:setText(g_i18n:getText("input_MoreDustExtension_toggle_exhaust"))
		toolTip:setText(g_i18n:getText("menu_MoreDustExtension_exhaust"))

		for optName, optL10n in pairs(MoreDustExtension.VEHICLE_L10N) do
			local fullName = "menuOption_" .. optName

			self[fullName]           = self.checkInvertYLook:clone()
			self[fullName]["target"] = g_moreDustExtension
			self[fullName]["id"]     = "moreDustExtension_" .. optName
			self[fullName]:setCallback("onClickCallback", "onMenuOptionChanged_ScaleChangePerc")
			self[fullName]:setDisabled(false)

			local settingTitle = self[fullName]["elements"][4]
			local toolTip      = self[fullName]["elements"][6]

			self[fullName]:setTexts(MoreDustExtension.MENU_PERCENT)

			settingTitle:setText(g_i18n:getText(optL10n[1]))
			toolTip:setText(g_i18n:getText("configuration_valueDefault") .. " : " ..( optL10n[2]*100 ) .. " %")
		end


		local title = TextElement.new()
		title:applyProfile("settingsMenuSubtitle", true)
		title:setText(g_i18n:getText("menu_MoreDustExtension_title"))

		self.boxLayout:addElement(title)
		self.boxLayout:addElement(self.menuOption_UserScale)
		for _, optName in ipairs(MoreDustExtension.VEHICLE_ORDER) do
			local thisOption = "menuOption_" .. optName
			self.boxLayout:addElement(self[thisOption])
		end
		self.boxLayout:addElement(self.menuOption_ToggleExhaust)
	end

	local thisModEnviroment = getfenv(0)["g_moreDustExtension"]

	self.menuOption_UserScale:setState(math.floor(thisModEnviroment.userScale * 10))
	self.menuOption_ToggleExhaust:setIsChecked(thisModEnviroment.TURN_OFF_EXHAUST)

	for optName, optCurrent in pairs(MoreDustExtension.VEHICLE_SCALE) do
		local thisMenuOption = "menuOption_" .. optName
		local thisOptionIDX  = math.floor(optCurrent * 20)
		self[thisMenuOption]:setState(thisOptionIDX)
	end
end

function MoreDustExtension:onMenuOptionChanged_toggleExhaust(state)
	local thisModEnviroment = getfenv(0)["g_moreDustExtension"]

	thisModEnviroment.TURN_OFF_EXHAUST = state == CheckedOptionElement.STATE_CHECKED
	thisModEnviroment:saveSettings()
end

function MoreDustExtension:onMenuOptionChanged_ScaleChangePerc(state, info)
	local thisModEnviroment = getfenv(0)["g_moreDustExtension"]
	local thisOption        = string.sub(info.id,19)

	thisModEnviroment.VEHICLE_SCALE[thisOption] = state / 20
	thisModEnviroment:saveSettings()
end

function MoreDustExtension:onMenuOptionChanged_ScaleChangeUser(state)
	local thisModEnviroment = getfenv(0)["g_moreDustExtension"]

	thisModEnviroment.userScale = state / 10
	thisModEnviroment:updateHelp()
	thisModEnviroment:saveSettings()
end
